import 'dart:io';

import 'package:admin_desktop/presentation/components/custom_line.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/booking/widgets/statistic_check_table.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/order_detail/print_page.dart';
import 'package:flutter/material.dart';
import 'package:flutter_esc_pos_utils/flutter_esc_pos_utils.dart';
import 'package:flutter_pos_printer_platform_image_3/flutter_pos_printer_platform_image_3.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import 'package:admin_desktop/presentation/components/components.dart';
import 'package:remixicon_updated/remixicon_updated.dart';
import '../../../../../../domain/models/response/booking_count_statistic_response.dart';
import 'package:admin_desktop/presentation/components/buttons/custom_mini_button.dart';
import 'package:admin_desktop/presentation/styles/style.dart';

class GenerateBookingStatisticCheck extends StatefulWidget {
  final BookingStatisticResponse? statistic;

  const GenerateBookingStatisticCheck({super.key, required this.statistic});

  @override
  State<GenerateBookingStatisticCheck> createState() =>
      _GenerateBookingStatisticCheckState();
}

class _GenerateBookingStatisticCheckState
    extends State<GenerateBookingStatisticCheck> {
  @override
  Widget build(BuildContext context) {
    return Container(
      padding: REdgeInsets.symmetric(horizontal: 14, vertical: 16),
      child: SingleChildScrollView(
        child: Column(
          mainAxisSize: MainAxisSize.min,
          crossAxisAlignment: CrossAxisAlignment.center,
          children: [
            Text(
                "${AppHelpers.getTranslation(TrKeys.booking)} ${AppHelpers.getTranslation(TrKeys.statistics)}",
                style: Style.interMedium(size: 20)),
            8.verticalSpace,
            Text(TimeService.dateFormatYMDHm(DateTime.now()),
                style: Style.interNormal(size: 14)),
            12.verticalSpace,
            CustomLine(),
            12.verticalSpace,
            Row(
              children: [
                Expanded(
                  child: Text(AppHelpers.getTranslation(TrKeys.shopName),
                      style: Style.interRegular(size: 14)),
                ),
                Text(
                  LocalStorage.getShop()?.translation?.title ?? "",
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                Expanded(
                  child: Text(
                    AppHelpers.getTranslation(TrKeys.totalCount),
                    style: Style.interRegular(size: 14),
                  ),
                ),
                Text(
                  "${widget.statistic?.totalCount ?? 0}",
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                Expanded(
                  child: Text(AppHelpers.getTranslation(TrKeys.totalPrice),
                      style: Style.interRegular(size: 14)),
                ),
                Text(
                  AppHelpers.numberFormat(number: widget.statistic?.totalPrice),
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            8.verticalSpace,
            const Divider(thickness: 2),
            10.verticalSpace,
            CustomLine(),
            12.verticalSpace,
            Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                Table(
                  columnWidths: {
                    0: FixedColumnWidth(80),
                    1: FixedColumnWidth(70),
                    2: FixedColumnWidth(70),
                    3: FixedColumnWidth(80),
                  },
                  children: [
                    TableRow(children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.status),
                        style: Style.interNormal(size: 14),
                      ),
                      Text(
                        AppHelpers.getTranslation(TrKeys.count),
                        style: Style.interNormal(size: 14),
                        textAlign: TextAlign.center,
                      ),
                      Text(
                        AppHelpers.getTranslation(TrKeys.percent),
                        style: Style.interNormal(size: 14),
                        textAlign: TextAlign.center,

                      ),
                      Text(
                        AppHelpers.getTranslation(TrKeys.price),
                        style: Style.interNormal(size: 14),
                        textAlign: TextAlign.end,
                      ),
                    ]),
                  ],
                ),
                10.verticalSpace,
                StatisticCheckTable(
                  status: TrKeys.newKey,
                  data: widget.statistic?.newBooked,
                ),
                StatisticCheckTable(
                  status: TrKeys.booked,
                  data: widget.statistic?.booked,
                ),
                StatisticCheckTable(
                  status: TrKeys.progress,
                  data: widget.statistic?.progress,
                ),
                StatisticCheckTable(
                  status: TrKeys.canceled,
                  data: widget.statistic?.canceled,
                ),
                StatisticCheckTable(
                  status: TrKeys.ended,
                  data: widget.statistic?.ended,
                ),
              ],
            ),
            12.verticalSpace,
            CustomLine(),
            20.verticalSpace,
            Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                Expanded(
                  child: LoginButton(
                      title: AppHelpers.getTranslation(TrKeys.print),
                      onPressed: () async {
                        if(LocalStorage.getPrinterData()==null){
                          final bytes = await _getBytes();
                          if (context.mounted) {
                            AppHelpers.showAlertDialog(
                                context: context, child: PrintPage(bytes: bytes));
                          }
                        }else {
                          final bytes = await _getBytes();
                          _printEscPos(
                            bytes,
                          );
                        }
                      }),
                ),
                10.horizontalSpace,
                CustomMiniButton(
                  size: 56,
                  onTap: () async {
                    final bytes = await _getBytes();
                    if (context.mounted) {
                      AppHelpers.showAlertDialog(
                          context: context, child: PrintPage(bytes: bytes));
                    }
                    // ignore: deprecated_member_use
                  },
                  icon: Remix.settings_5_fill,
                  iconColor: Style.white.withOpacity(0.7),
                  color: Style.primary,
                )
              ],
            ),
          ],
        ),
      ),
    );
  }

  var printerManager = PrinterManager.instance;

  List<int>? pendingTask;

  final _reconnect = false;

  final BTStatus _currentStatus = BTStatus.none;

  void _printEscPos(List<int> bytes,) async {
    final profile = await CapabilityProfile.load();
    final generator = Generator(PaperSize.mm58, profile);
    var connectedTCP = false;
    if (LocalStorage.getPrinterData() == null) return;
    var bluetoothPrinter = LocalStorage.getPrinterData()!;

    switch (bluetoothPrinter.typePrinter) {
      case PrinterType.usb:
        debugPrint("");
        bytes += generator.feed(2);
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: UsbPrinterInput(
                name: bluetoothPrinter.deviceName,
                productId: bluetoothPrinter.productId,
                vendorId: bluetoothPrinter.vendorId));
        pendingTask = null;
        break;
      case PrinterType.bluetooth:
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: BluetoothPrinterInput(
                name: bluetoothPrinter.deviceName,
                address: bluetoothPrinter.address!,
                isBle: bluetoothPrinter.isBle ?? false,
                autoConnect: _reconnect));
        pendingTask = null;
        if (Platform.isAndroid) pendingTask = bytes;
        break;
      case PrinterType.network:
        bytes += generator.feed(2);
        bytes += generator.cut();
        connectedTCP = await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: TcpPrinterInput(ipAddress: bluetoothPrinter.address!));
        if (!connectedTCP) debugPrint(' --- please review your connection ---');
        break;
    }
    if (bluetoothPrinter.typePrinter == PrinterType.bluetooth &&
        Platform.isAndroid) {
      if (_currentStatus == BTStatus.connected) {
        printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
        pendingTask = null;
      }
    } else {
      printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
    }
  }


  Future<List<int>> _getBytes() async {
    String customDivider = "-------------------------------------------";
    String customLine = "___________________________________________";
    String customSpace = "  ";
    List<int> bytes = [];
    final profile = await CapabilityProfile.load(name: 'XP-N160I');
    //     // PaperSize.mm80 or PaperSize.mm58
    final generator = Generator(PaperSize.mm80, profile);

    bytes += generator.setGlobalCodeTable('CP1252');
    bytes += generator.setStyles(const PosStyles(align: PosAlign.center));
    bytes += generator.text(
        "${AppHelpers.getTranslation(TrKeys.booking)} ${AppHelpers.getTranslation(TrKeys.statistics)}",
        styles: const PosStyles(align: PosAlign.center, bold: true));

    bytes += generator.text(TimeService.dateFormatYMDHm(DateTime.now()),
        styles: const PosStyles(align: PosAlign.center));
    bytes += generator.text(customDivider);
    bytes += generator.row([
      PosColumn(
          width: 7,
          text: customSpace + AppHelpers.getTranslation(TrKeys.shopName),
          styles: const PosStyles(align: PosAlign.left, codeTable: 'CP1252')),
      PosColumn(
          width: 5,
          text: LocalStorage.getShop()?.translation?.title ?? "",
          styles: const PosStyles(align: PosAlign.right, codeTable: 'CP1252')),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.totalCount),
        styles: const PosStyles(align: PosAlign.left),
      ),
      PosColumn(
        width: 6,
        text: "${widget.statistic?.totalCount ?? 0}",
        styles: const PosStyles(align: PosAlign.right),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.totalPrice),
        styles: const PosStyles(
          align: PosAlign.left,
        ),
      ),
      PosColumn(
        width: 6,
        text: "${widget.statistic?.totalPrice ?? 0}",
        styles: const PosStyles(
          align: PosAlign.right,
        ),
      ),
    ]);

    bytes += generator.text(customLine);

    bytes += generator.text(customDivider);

    bytes += generator.row([
      PosColumn(
          text: customSpace + AppHelpers.getTranslation(TrKeys.status),
          width: 4,
          styles: PosStyles(bold: true, align: PosAlign.left)),
      PosColumn(
          text: AppHelpers.getTranslation(TrKeys.count),
          width: 2,
          styles: PosStyles(bold: true, align: PosAlign.left)),
      PosColumn(
          text: AppHelpers.getTranslation(TrKeys.percent),
          width: 2,
          styles: PosStyles(bold: true, align: PosAlign.left)),
      PosColumn(
          text: AppHelpers.getTranslation(TrKeys.price),
          width: 4,
          styles: PosStyles(bold: true, align: PosAlign.right)),
    ]);
    bytes += generator.text(customDivider);

    bytes += await createTable(
        status: TrKeys.newKey, data: widget.statistic?.newBooked);
    bytes += await createTable(
        status: TrKeys.booked, data: widget.statistic?.booked);
    bytes += await createTable(
        status: TrKeys.progress, data: widget.statistic?.progress);
    bytes += await createTable(
        status: TrKeys.canceled, data: widget.statistic?.canceled);
    bytes +=
        await createTable(status: TrKeys.ended, data: widget.statistic?.ended);
    bytes += generator.text(customDivider);
    bytes += generator.text(AppHelpers.getTranslation(TrKeys.thankYou));
    return bytes;
  }

  Future<List<int>> createTable(
      {required String status, required Booked? data}) async {
    List<int> bytes = [];
    final profile = await CapabilityProfile.load(name: 'XP-N160I');
    String customSpace = "  ";
    final generator = Generator(PaperSize.mm80, profile);

    bytes += generator.row([
      PosColumn(
          text: customSpace + AppHelpers.getTranslation(status),
          width: 4,
          styles: PosStyles(align: PosAlign.left)),
      PosColumn(
          text: "${data?.count ?? 0}",
          width: 2,
          styles: PosStyles(align: PosAlign.left)),
      PosColumn(
          text: "${data?.percent ?? 0}",
          width: 2,
          styles: PosStyles(align: PosAlign.left)),
      PosColumn(
          text: "${data?.totalPrice ?? 0}",
          width: 4,
          styles: PosStyles(align: PosAlign.right)),
    ]);

    return bytes;
  }
}
